/**
 * Use this file for JavaScript code that you want to run in the front-end
 * on posts/pages that contain this block.
 */

document.addEventListener('DOMContentLoaded', function() {
	const navContainer = document.querySelector('.in-page-nav-menu-container');
	
	if (!navContainer) {
		return;
	}

	const navList = navContainer.querySelector('.in-page-nav-menu-list');
	const toggleButton = navContainer.querySelector('.in-page-nav-menu-toggle');
	const toggleText = toggleButton?.querySelector('.in-page-nav-menu-toggle-text');
	
	// Find the main content area - try common selectors for WordPress themes
	const contentSelectors = [
		'main',
		'article',
		'.entry-content',
		'.post-content',
		'.content',
		'[role="main"]',
		'.site-content',
		'#content',
		'.wp-block-post-content'
	];
	
	let contentArea = null;
	for (const selector of contentSelectors) {
		contentArea = document.querySelector(selector);
		if (contentArea) {
			break;
		}
	}
	
	// If no content area found, fall back to body but exclude header/footer
	if (!contentArea) {
		contentArea = document.body;
	}
	
	// Find all headings within the content area (H2-H6), excluding those inside the nav block itself
	const allHeadings = contentArea.querySelectorAll('h2, h3, h4, h5, h6');
	const headings = Array.from(allHeadings).filter(heading => {
		// Exclude headings that are inside the navigation block
		if (navContainer.contains(heading)) {
			return false;
		}
		
		// Exclude headings that are in header, footer, or sidebar elements
		const parent = heading.closest('header, footer, aside, nav:not(.in-page-nav-menu-container), .sidebar, .widget, [role="banner"], [role="contentinfo"], [role="complementary"], [role="navigation"]');
		if (parent && !parent.classList.contains('in-page-nav-menu-container')) {
			return false;
		}
		
		return true;
	});
	
	if (headings.length === 0) {
		navContainer.style.display = 'none';
		return;
	}

	// Ensure all headings have IDs and are focusable
	let idCounter = 0;
	const usedIds = new Set();
	
	headings.forEach((heading, index) => {
		if (!heading.id) {
			let baseId = heading.textContent.trim()
				.toLowerCase()
				.replace(/[^a-z0-9]+/g, '-')
				.replace(/^-+|-+$/g, '');
			
			if (!baseId) {
				baseId = 'heading-' + idCounter;
				idCounter++;
			}
			
			// Ensure unique ID
			let finalId = baseId;
			let counter = 1;
			while (usedIds.has(finalId) || document.getElementById(finalId)) {
				finalId = baseId + '-' + counter;
				counter++;
			}
			
			heading.id = finalId;
			usedIds.add(finalId);
		}
		
		if (!heading.hasAttribute('tabindex')) {
			heading.setAttribute('tabindex', '-1');
		}
	});

	// Build navigation items
	navList.innerHTML = '';
	headings.forEach((heading, index) => {
		const li = document.createElement('li');
		li.className = 'in-page-nav-menu-item';
		
		const link = document.createElement('a');
		link.className = 'in-page-nav-menu-link';
		link.href = '#' + heading.id;
		link.setAttribute('data-target', heading.id);
		link.setAttribute('aria-label', 
			`Jump to section ${index + 1} of ${headings.length}: ${heading.textContent.trim()}`);
		
		const line = document.createElement('span');
		line.className = 'in-page-nav-menu-line';
		line.setAttribute('aria-hidden', 'true');
		
		const text = document.createElement('span');
		text.className = 'in-page-nav-menu-text';
		text.textContent = heading.textContent.trim();
		
		link.appendChild(line);
		link.appendChild(text);
		li.appendChild(link);
		navList.appendChild(li);
	});

	// Update toggle text with count
	if (toggleText) {
		toggleText.textContent = `Jump to section (${headings.length} items)`;
	}

	// Mobile toggle functionality
	if (toggleButton) {
		toggleButton.addEventListener('click', function() {
			const isExpanded = navContainer.getAttribute('aria-expanded') === 'true';
			navContainer.setAttribute('aria-expanded', !isExpanded);
		});
	}

	// Get all nav links after they've been created
	const navLinks = navList.querySelectorAll('.in-page-nav-menu-link');

	// Smooth scroll behavior and focus management
	navLinks.forEach(link => {
		link.addEventListener('click', function(e) {
			e.preventDefault();
			const targetId = this.getAttribute('data-target');
			const targetElement = document.getElementById(targetId);
			
			if (targetElement) {
				// Smooth scroll to target
				targetElement.scrollIntoView({
					behavior: 'smooth',
					block: 'start'
				});
				
				// Wait for scroll to complete, then shift focus
				setTimeout(function() {
					targetElement.focus();
					
					// Add visual focus indicator
					targetElement.style.outline = '2px solid currentColor';
					targetElement.style.outlineOffset = '4px';
					
					// Remove outline on blur or after delay
					const removeOutline = function() {
						targetElement.style.outline = '';
						targetElement.style.outlineOffset = '';
					};
					
					targetElement.addEventListener('blur', removeOutline, { once: true });
					setTimeout(removeOutline, 3000);
				}, 500);
				
				// Collapse mobile menu after selection
				if (window.innerWidth < 768 || window.innerHeight < 600) {
					navContainer.setAttribute('aria-expanded', 'false');
				}
			}
		});
		
		// Keyboard navigation: Enter and Space to activate
		link.addEventListener('keydown', function(e) {
			if (e.key === 'Enter' || e.key === ' ') {
				e.preventDefault();
				this.click();
			}
		});
	});

	// Active state on scroll
	let ticking = false;
	function updateActiveItem() {
		const scrollPosition = window.scrollY + 100;
		
		let activeHeading = null;
		headings.forEach(heading => {
			if (heading.offsetTop <= scrollPosition) {
				activeHeading = heading;
			}
		});

		navLinks.forEach(link => {
			link.classList.remove('active');
			link.removeAttribute('aria-current');
			if (activeHeading && link.getAttribute('data-target') === activeHeading.id) {
				link.classList.add('active');
				link.setAttribute('aria-current', 'location');
			}
		});

		ticking = false;
	}

	window.addEventListener('scroll', function() {
		if (!ticking) {
			window.requestAnimationFrame(updateActiveItem);
			ticking = true;
		}
	});

	// Initial check
	updateActiveItem();
});