
/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, InspectorControls } from '@wordpress/block-editor';
import { PanelBody, RangeControl, ColorPalette } from '@wordpress/components';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @param {Object} props Block properties.
 * @return {Element} Element to render.
 */
export default function Edit( { attributes, setAttributes } ) {
	const {
		lineColor,
		lineColorHover,
		textColor,
		lineThickness,
		lineThicknessHover,
		lineLength,
		lineLengthHover,
	} = attributes;

	const colors = [
		{ name: 'Black', color: '#000000' },
		{ name: 'White', color: '#ffffff' },
		{ name: 'Blue', color: '#0073aa' },
		{ name: 'Red', color: '#dc3232' },
		{ name: 'Green', color: '#46b450' },
		{ name: 'Orange', color: '#f56e28' },
	];

	return (
		<>
			<InspectorControls>
				<PanelBody title={ __( 'Line Settings', 'in-page-nav-menu' ) }>
					<p>
						<strong>
							{ __( 'Default Line Color', 'in-page-nav-menu' ) }
						</strong>
					</p>
					<ColorPalette
						colors={ colors }
						value={ lineColor }
						onChange={ ( color ) =>
							setAttributes( { lineColor: color } )
						}
					/>

					<p>
						<strong>
							{ __( 'Hover Line Color', 'in-page-nav-menu' ) }
						</strong>
					</p>
					<ColorPalette
						colors={ colors }
						value={ lineColorHover }
						onChange={ ( color ) =>
							setAttributes( { lineColorHover: color } )
						}
					/>

					<RangeControl
						label={ __( 'Line Thickness', 'in-page-nav-menu' ) }
						value={ lineThickness }
						onChange={ ( value ) =>
							setAttributes( { lineThickness: value } )
						}
						min={ 1 }
						max={ 8 }
					/>

					<RangeControl
						label={ __(
							'Line Thickness (Hover)',
							'in-page-nav-menu'
						) }
						value={ lineThicknessHover }
						onChange={ ( value ) =>
							setAttributes( { lineThicknessHover: value } )
						}
						min={ 1 }
						max={ 10 }
					/>

					<RangeControl
						label={ __( 'Line Length (px)', 'in-page-nav-menu' ) }
						value={ lineLength }
						onChange={ ( value ) =>
							setAttributes( { lineLength: value } )
						}
						min={ 20 }
						max={ 80 }
					/>

					<RangeControl
						label={ __(
							'Line Length Hover (px)',
							'in-page-nav-menu'
						) }
						value={ lineLengthHover }
						onChange={ ( value ) =>
							setAttributes( { lineLengthHover: value } )
						}
						min={ 40 }
						max={ 120 }
					/>
				</PanelBody>

				<PanelBody
					title={ __( 'Text Settings', 'in-page-nav-menu' ) }
					initialOpen={ false }
				>
					<p>
						<strong>
							{ __( 'Text Color', 'in-page-nav-menu' ) }
						</strong>
					</p>
					<ColorPalette
						colors={ colors }
						value={ textColor }
						onChange={ ( color ) =>
							setAttributes( { textColor: color } )
						}
					/>
				</PanelBody>
			</InspectorControls>

			<div { ...useBlockProps() }>
				<div className="in-page-nav-menu-preview">
					<div className="in-page-nav-menu-icon">
						<svg
							width="60"
							height="60"
							viewBox="0 0 60 60"
							fill="none"
						>
							<line
								x1="10"
								y1="15"
								x2="40"
								y2="15"
								stroke={ lineColor }
								strokeWidth={ lineThickness }
								strokeLinecap="round"
							/>
							<line
								x1="10"
								y1="25"
								x2="40"
								y2="25"
								stroke={ lineColor }
								strokeWidth={ lineThickness }
								strokeLinecap="round"
							/>
							<line
								x1="10"
								y1="35"
								x2="40"
								y2="35"
								stroke={ lineColor }
								strokeWidth={ lineThickness }
								strokeLinecap="round"
							/>
							<line
								x1="10"
								y1="45"
								x2="40"
								y2="45"
								stroke={ lineColor }
								strokeWidth={ lineThickness }
								strokeLinecap="round"
							/>
						</svg>
					</div>
					<div className="in-page-nav-menu-description">
						<h3>
							{ __(
								'In-Page Navigation Menu',
								'in-page-nav-menu'
							) }
						</h3>
						<p>
							{ __(
								'This block will create an interactive table of contents anchored to the left side of the viewport. It will automatically detect headings on the page.',
								'in-page-nav-menu'
							) }
						</p>
						<p className="in-page-nav-menu-note">
							{ __(
								'Note: Only visible on tablet and desktop screens (768px+)',
								'in-page-nav-menu'
							) }
						</p>
					</div>
				</div>
			</div>
		</>
	);
}
